LIBRARY ieee;
   use ieee.std_logic_1164.all;

package PCpuRegistersTX is

   -- Cpu control registers (values are changed by uC, can be read by uC and VHDL)
   type TCpuCtrlRegEnum is
   (
      -- 0 CPU

      REG_CTRL_FIRMWARE_VERSION_PROD,
      REG_CTRL_FIRMWARE_VERSION_MAJ,
      REG_CTRL_FIRMWARE_VERSION_MIN,

      REG_CTRL_MASK_LEVEL_INT,
      REG_CTRL_MASK_EDGE_INT,

      REG_CTRL_DEBUG,

      REG_CTRL_TIMER_0_HL,
      REG_CTRL_TIMER_0_LH,
      REG_CTRL_TIMER_0_LL,

      REG_CTRL_TIMER_1_HL,
      REG_CTRL_TIMER_1_LH,
      REG_CTRL_TIMER_1_LL,

      REG_CTRL_USB,
      REG_CTRL_USB_PROD_ID,
      REG_CTRL_USB_STR_IDX,
      REG_CTRL_USB_SERIAL_7,
      REG_CTRL_USB_SERIAL_6,
      REG_CTRL_USB_SERIAL_5,
      REG_CTRL_USB_SERIAL_4,
      REG_CTRL_USB_SERIAL_3,
      REG_CTRL_USB_SERIAL_2,
      REG_CTRL_USB_SERIAL_1,
      REG_CTRL_USB_SERIAL_0,
      REG_CTRL_USB_BUF_ADD,
      REG_CTRL_USB_BUF_DATA,
      REG_CTRL_USB_BUF_WE,

      -- icap control register
      REG_CTRL_ICAP_SEND,
      REG_CTRL_ICAP_OPCODE_LSB,
      REG_CTRL_ICAP_OPCODE_MSB,
      REG_CTRL_ICAP_USERDATA_LSB,
      REG_CTRL_ICAP_USERDATA_MSB,
      REG_CTRL_ICAP_FALLBACK,

      -- 1 Communication with on board chips (SPI, I2C)

      REG_CTRL_SPI,
      REG_CTRL_SPI_REQUEST,
      REG_CTRL_SPI_CMD,
      REG_CTRL_SPI_SRC_START_ADD_HL,
      REG_CTRL_SPI_SRC_START_ADD_LH,
      REG_CTRL_SPI_SRC_START_ADD_LL,
      REG_CTRL_SPI_SRC_END_ADD_HL,
      REG_CTRL_SPI_SRC_END_ADD_LH,
      REG_CTRL_SPI_SRC_END_ADD_LL,
      REG_CTRL_SPI_DEST_START_ADD_HL,
      REG_CTRL_SPI_DEST_START_ADD_LH,
      REG_CTRL_SPI_DEST_START_ADD_LL,
      REG_CTRL_SPI_LENGTH,
      REG_CTRL_SPI_FIFO,

      REG_CTRL_I2C_CFG1,
      REG_CTRL_I2C_PLL,

      REG_CTRL_I2C_DEV_CFG1,
      REG_CTRL_I2C_DEV_PLL,

      REG_CTRL_I2C_REG_CFG1,
      REG_CTRL_I2C_REG_PLL,

      REG_CTRL_I2C_LENGTH_CFG1,
      REG_CTRL_I2C_LENGTH_PLL,

      REG_CTRL_I2C_FIFO_CFG1,
      REG_CTRL_I2C_FIFO_PLL,

      -- 2 Misc

      REG_CTRL_RST,

      REG_CTRL_EDID_ADD,
      REG_CTRL_EDID_DATA,
      REG_CTRL_EDID_MISC,

      REG_CTRL_GEN1,

      -- 3 Video

      REG_CTRL_H_TOTAL_H,
      REG_CTRL_H_TOTAL_L,
      REG_CTRL_H_ACTIVE_H,
      REG_CTRL_H_ACTIVE_L,
      REG_CTRL_H_BP_H,
      REG_CTRL_H_BP_L,
      REG_CTRL_H_SYNC_WIDTH_H,
      REG_CTRL_H_SYNC_WIDTH_L,
      REG_CTRL_V_TOTAL_H,
      REG_CTRL_V_TOTAL_L,
      REG_CTRL_V_ACTIVE_H,
      REG_CTRL_V_ACTIVE_L,
      REG_CTRL_V_BP,
      REG_CTRL_V_SYNC_WIDTH,
      REG_CTRL_H_FREQ_H,
      REG_CTRL_H_FREQ_L,
      REG_CTRL_V_FREQ_H,
      REG_CTRL_V_FREQ_L,
      REG_CTRL_VIDEO_MISC,
      REG_CTRL_VIDEO_MODE,

      REG_CTRL_CRC,

      REG_CTRL_VIDEO_SOURCE_1,
      REG_CTRL_VIDEO_SOURCE_2,
      REG_CTRL_VIDEO_SOURCE_3,
      REG_CTRL_VIDEO_SOURCE_4,
      REG_CTRL_VIDEO_SOURCE_5,
      REG_CTRL_VIDEO_SOURCE_6,

      -- 4 HDCP

      REG_CTRL_HDCP_CPBL,

      REG_CTRL_HDCP_BCAPS_PORT1,

      REG_CTRL_KSV_RAM_ADD,
      REG_CTRL_KSV_RAM_WE,
      REG_CTRL_KSV_RAM_DATA,

      REG_CTRL_NEW_BKSV,

      REG_CTRL_HDCP_IN_ENABLED,

      REG_CTRL_TOGGLE_HPD_IN,

      -- 5 Communication with Packet Router

      REG_CTRL_TX_PACKET_7,
      REG_CTRL_TX_PACKET_6,
      REG_CTRL_TX_PACKET_5,
      REG_CTRL_TX_PACKET_4,
      REG_CTRL_TX_PACKET_3,
      REG_CTRL_TX_PACKET_2,
      REG_CTRL_TX_PACKET_1,
      REG_CTRL_TX_PACKET_0,
      REG_CTRL_TX_PACKET_ADD_CAT,
      REG_CTRL_TX_PACKET_GEN,

      REG_CTRL_RX_PACKET_GEN_GTP0,
      REG_CTRL_RX_PACKET_GEN_GTP1,
      REG_CTRL_RX_PACKET_GEN_GTP2,
      REG_CTRL_RX_PACKET_GEN_GTP3,

      -- 6 Packet Router

      REG_CTRL_ROUTER_TESTER_A_TX_1,
      REG_CTRL_ROUTER_TESTER_A_TX_2,
      REG_CTRL_ROUTER_TESTER_A_TX_3,
      REG_CTRL_ROUTER_TESTER_A_TX_4_LL,
      REG_CTRL_ROUTER_TESTER_A_TX_4_LH,
      REG_CTRL_ROUTER_TESTER_A_TX_4_HL,
      REG_CTRL_ROUTER_TESTER_A_TX_4_HH,
      REG_CTRL_ROUTER_TESTER_A_RX_1,

      REG_CTRL_ROUTER_TESTER_B_TX_1,
      REG_CTRL_ROUTER_TESTER_B_TX_2,
      REG_CTRL_ROUTER_TESTER_B_TX_3,
      REG_CTRL_ROUTER_TESTER_B_TX_4_LL,
      REG_CTRL_ROUTER_TESTER_B_TX_4_LH,
      REG_CTRL_ROUTER_TESTER_B_TX_4_HL,
      REG_CTRL_ROUTER_TESTER_B_TX_4_HH,
      REG_CTRL_ROUTER_TESTER_B_RX_1,

      REG_CTRL_ROUTER_BRIDGE_0_GENERAL_1,
      REG_CTRL_ROUTER_BRIDGE_0_CHANNEL_LL,
      REG_CTRL_ROUTER_BRIDGE_0_CHANNEL_LH,
      REG_CTRL_ROUTER_BRIDGE_0_CHANNEL_HL,
      REG_CTRL_ROUTER_BRIDGE_0_CHANNEL_HH,

      REG_CTRL_ROUTER_BRIDGE_1_GENERAL_1,
      REG_CTRL_ROUTER_BRIDGE_1_CHANNEL_LL,
      REG_CTRL_ROUTER_BRIDGE_1_CHANNEL_LH,
      REG_CTRL_ROUTER_BRIDGE_1_CHANNEL_HL,
      REG_CTRL_ROUTER_BRIDGE_1_CHANNEL_HH,

      REG_CTRL_ROUTER_BRIDGE_2_GENERAL_1,
      REG_CTRL_ROUTER_BRIDGE_2_CHANNEL_LL,
      REG_CTRL_ROUTER_BRIDGE_2_CHANNEL_LH,
      REG_CTRL_ROUTER_BRIDGE_2_CHANNEL_HL,
      REG_CTRL_ROUTER_BRIDGE_2_CHANNEL_HH,

      REG_CTRL_ROUTER_BRIDGE_3_GENERAL_1,
      REG_CTRL_ROUTER_BRIDGE_3_CHANNEL_LL,
      REG_CTRL_ROUTER_BRIDGE_3_CHANNEL_LH,
      REG_CTRL_ROUTER_BRIDGE_3_CHANNEL_HL,
      REG_CTRL_ROUTER_BRIDGE_3_CHANNEL_HH,

      -- 7 Audio
      
      REG_CTRL_AUDIO_NOT_USED0,
      REG_CTRL_AUDIO_NOT_USED1,
      REG_CTRL_AUDIO_AUX_DAC_DATA,
      REG_CTRL_AUDIO_AUX_DAC_ADDR,
      REG_CTRL_AUDIO_AUX_DAC_WE
   );

   -- Cpu status registers (values are changed by VHDL, can be read by uC and VHDL)
   type TCpuStatRegEnum is
   (
      -- 0 CPU

      REG_STAT_LEVEL_INT_CPU,

      REG_STAT_USB,
      REG_STAT_USB_BUF_DATA,

      REG_STAT_CPU_ROM_EXTRA_DATA_PATCH,
      REG_STAT_FPGA_BUILD_NUMBER_LSB,
      REG_STAT_FPGA_BUILD_NUMBER_MSB,

      REG_STAT_ICAP_BUSY,
      REG_STAT_ICAP_USERDATA_LSB,
      REG_STAT_ICAP_USERDATA_MSB,

      -- 1 Communication with on board chips (SPI, I2C)

      REG_STAT_SPI_FIFO,
      REG_STAT_SPI_CHECKSUM,

      REG_STAT_I2C_FIFO_CFG1,
      REG_STAT_I2C_FIFO_PLL,

      -- 2 Misc

      REG_STAT_LP_HPD,

      REG_STAT_EDID_DATA,

      REG_STAT_GEN1,

      REG_STAT_BOARD_ID,

      -- 3 Video

      REG_STAT_H_TOTAL_H,
      REG_STAT_H_TOTAL_L,
      REG_STAT_H_ACTIVE_H,
      REG_STAT_H_ACTIVE_L,
      REG_STAT_H_BP_H,
      REG_STAT_H_BP_L,
      REG_STAT_H_SYNC_WIDTH_H,
      REG_STAT_H_SYNC_WIDTH_L,
      REG_STAT_V_TOTAL_H,
      REG_STAT_V_TOTAL_L,
      REG_STAT_V_ACTIVE_H,
      REG_STAT_V_ACTIVE_L,
      REG_STAT_V_BP,
      REG_STAT_V_SYNC_WIDTH,
      REG_STAT_H_FREQ_H,
      REG_STAT_H_FREQ_L,
      REG_STAT_V_FREQ_H,
      REG_STAT_V_FREQ_L,
      REG_STAT_VIDEO_MISC,

      REG_STAT_CRC_DVO_ERR_CNT_LL,
      REG_STAT_CRC_DVO_ERR_CNT_LH,
      REG_STAT_CRC_DVO_ERR_CNT_HL,
      REG_STAT_CRC_DVO_ERR_CNT_HH,

      REG_STAT_CRC_WRITE_ERR_CNT_LL,
      REG_STAT_CRC_WRITE_ERR_CNT_LH,
      REG_STAT_CRC_WRITE_ERR_CNT_HL,
      REG_STAT_CRC_WRITE_ERR_CNT_HH,

      REG_STAT_CRC_READ_ERR_CNT_LL,
      REG_STAT_CRC_READ_ERR_CNT_LH,
      REG_STAT_CRC_READ_ERR_CNT_HL,
      REG_STAT_CRC_READ_ERR_CNT_HH,

      REG_STAT_CRC_DVO_FRM_CNT_LL,
      REG_STAT_CRC_DVO_FRM_CNT_LH,
      REG_STAT_CRC_DVO_FRM_CNT_HL,
      REG_STAT_CRC_DVO_FRM_CNT_HH,

      REG_STAT_CRC_WRITE_FRM_CNT_LL,
      REG_STAT_CRC_WRITE_FRM_CNT_LH,
      REG_STAT_CRC_WRITE_FRM_CNT_HL,
      REG_STAT_CRC_WRITE_FRM_CNT_HH,

      REG_STAT_CRC_READ_FRM_CNT_LL,
      REG_STAT_CRC_READ_FRM_CNT_LH,
      REG_STAT_CRC_READ_FRM_CNT_HL,
      REG_STAT_CRC_READ_FRM_CNT_HH,

      REG_STAT_CRC_VAL_LL,
      REG_STAT_CRC_VAL_LH,
      REG_STAT_CRC_VAL_HL,
      REG_STAT_CRC_VAL_HH,

      -- 4 HDCP

      REG_STAT_KSV_RAM_DATA,

      -- 5 Communication with Packet Router

      REG_STAT_TX_PACKET_GEN,

      REG_STAT_RX_PACKET_7_GTP0,
      REG_STAT_RX_PACKET_7_GTP1,
      REG_STAT_RX_PACKET_7_GTP2,
      REG_STAT_RX_PACKET_7_GTP3,
      REG_STAT_RX_PACKET_6_GTP0,
      REG_STAT_RX_PACKET_6_GTP1,
      REG_STAT_RX_PACKET_6_GTP2,
      REG_STAT_RX_PACKET_6_GTP3,
      REG_STAT_RX_PACKET_5_GTP0,
      REG_STAT_RX_PACKET_5_GTP1,
      REG_STAT_RX_PACKET_5_GTP2,
      REG_STAT_RX_PACKET_5_GTP3,
      REG_STAT_RX_PACKET_4_GTP0,
      REG_STAT_RX_PACKET_4_GTP1,
      REG_STAT_RX_PACKET_4_GTP2,
      REG_STAT_RX_PACKET_4_GTP3,
      REG_STAT_RX_PACKET_3_GTP0,
      REG_STAT_RX_PACKET_3_GTP1,
      REG_STAT_RX_PACKET_3_GTP2,
      REG_STAT_RX_PACKET_3_GTP3,
      REG_STAT_RX_PACKET_2_GTP0,
      REG_STAT_RX_PACKET_2_GTP1,
      REG_STAT_RX_PACKET_2_GTP2,
      REG_STAT_RX_PACKET_2_GTP3,
      REG_STAT_RX_PACKET_1_GTP0,
      REG_STAT_RX_PACKET_1_GTP1,
      REG_STAT_RX_PACKET_1_GTP2,
      REG_STAT_RX_PACKET_1_GTP3,
      REG_STAT_RX_PACKET_0_GTP0,
      REG_STAT_RX_PACKET_0_GTP1,
      REG_STAT_RX_PACKET_0_GTP2,
      REG_STAT_RX_PACKET_0_GTP3,
      REG_STAT_RX_PACKET_ADD_CAT_GTP0,
      REG_STAT_RX_PACKET_ADD_CAT_GTP1,
      REG_STAT_RX_PACKET_ADD_CAT_GTP2,
      REG_STAT_RX_PACKET_ADD_CAT_GTP3,
      REG_STAT_RX_PACKET_GEN_GTP0,
      REG_STAT_RX_PACKET_GEN_GTP1,
      REG_STAT_RX_PACKET_GEN_GTP2,
      REG_STAT_RX_PACKET_GEN_GTP3,

      -- 6 Packet Router

      REG_STAT_ROUTER_TESTER_A_FRAME_COUNT_LL,
      REG_STAT_ROUTER_TESTER_A_FRAME_COUNT_LH,
      REG_STAT_ROUTER_TESTER_A_FRAME_COUNT_HL,
      REG_STAT_ROUTER_TESTER_A_FRAME_COUNT_HH,

      REG_STAT_ROUTER_TESTER_A_ERROR_COUNT_LL,
      REG_STAT_ROUTER_TESTER_A_ERROR_COUNT_LH,
      REG_STAT_ROUTER_TESTER_A_ERROR_COUNT_HL,
      REG_STAT_ROUTER_TESTER_A_ERROR_COUNT_HH,

      REG_STAT_ROUTER_TESTER_A_CRC_LL,
      REG_STAT_ROUTER_TESTER_A_CRC_LH,
      REG_STAT_ROUTER_TESTER_A_CRC_HL,
      REG_STAT_ROUTER_TESTER_A_CRC_HH,

      REG_STAT_ROUTER_TESTER_B_FRAME_COUNT_LL,
      REG_STAT_ROUTER_TESTER_B_FRAME_COUNT_LH,
      REG_STAT_ROUTER_TESTER_B_FRAME_COUNT_HL,
      REG_STAT_ROUTER_TESTER_B_FRAME_COUNT_HH,

      REG_STAT_ROUTER_TESTER_B_ERROR_COUNT_LL,
      REG_STAT_ROUTER_TESTER_B_ERROR_COUNT_LH,
      REG_STAT_ROUTER_TESTER_B_ERROR_COUNT_HL,
      REG_STAT_ROUTER_TESTER_B_ERROR_COUNT_HH,

      REG_STAT_ROUTER_TESTER_B_CRC_LL,
      REG_STAT_ROUTER_TESTER_B_CRC_LH,
      REG_STAT_ROUTER_TESTER_B_CRC_HL,
      REG_STAT_ROUTER_TESTER_B_CRC_HH
   );

   -- Cpu interrupt registers (Set to 1 if rising edge in VHDL, clear to 0 if uC writes 0)
   type TCpuIntRegEnum is
   (
      -- 0 CPU

      REG_INT_EDGE_INT_CPU,

      REG_INT_MISC,

      REG_INT_EDGE_INT_CPU_OCCURED,
      REG_INT_LEVEL_INT_CPU_OCCURED,

      -- 1 Communication with on board chips (SPI, I2C)

      REG_INT_SPI,

      REG_INT_I2C_CFG1,
      REG_INT_I2C_PLL
   );

   type TRegAddInit is record
     vAdd         : std_logic_vector(15 downto 0);
     vInitVal     : std_logic_vector(7 downto 0);
   end record;

   type TCpuCtrlRegDataArray     is array (TCpuCtrlRegEnum) of std_logic_vector(7 downto 0);
   type TCpuCtrlRegAddInitArray  is array (TCpuCtrlRegEnum) of TRegAddInit;

   type TCpuStatRegDataArray     is array (TCpuStatRegEnum) of std_logic_vector(7 downto 0);
   type TCpuStatRegAddArray      is array (TCpuStatRegEnum) of std_logic_vector(15 downto 0);

   type TCpuIntRegDataArray      is array (TCpuIntRegEnum)  of std_logic_vector(7 downto 0);
   type TCpuIntRegAddArray       is array (TCpuIntRegEnum)  of std_logic_vector(15 downto 0);

   -- Cpu control registers address
   constant CTRL_REG_ADD_INIT : TCpuCtrlRegAddInitArray :=
   (
      -- ADD      INIT

      -- 0 CPU

      (x"0000",   x"00"),  -- REG_CTRL_FIRMWARE_VERSION_PROD
      (x"0001",   x"00"),  -- REG_CTRL_FIRMWARE_VERSION_MAJ
      (x"0002",   x"00"),  -- REG_CTRL_FIRMWARE_VERSION_MIN

      (x"0003",   x"00"),  -- REG_CTRL_MASK_LEVEL_INT
      (x"0004",   x"00"),  -- REG_CTRL_MASK_EDGE_INT

      (x"0005",   x"F0"),  -- REG_CTRL_DEBUG
                              -- For debug purpose

      (x"0006",   x"00"),  -- REG_CTRL_TIMER_0_HL (used for I2C transactions)
                              -- Timer control and highest bit of load value
                              -- Bit 0:      Highest bit of timer load value
                              -- Bit 1:      Unused
                              -- Bit 2:      Unused
                              -- Bit 3:      Unused
                              -- Bit 4:      Unused
                              -- Bit 5:      Unused
                              -- Bit 6:      Load value in timer counter
                              -- Bit 7:      Enable timer count
      (x"0007",   x"00"),  -- REG_CTRL_TIMER_0_LH
                              -- Timer load value HL
      (x"0008",   x"00"),  -- REG_CTRL_TIMER_0_LL
                              -- Timer load value LL

      (x"0009",   x"00"),  -- REG_CTRL_TIMER_1_HL (used for TX DDC transactions)
      (x"000A",   x"00"),  -- REG_CTRL_TIMER_1_LH
      (x"000B",   x"00"),  -- REG_CTRL_TIMER_1_LL

      (x"000C",   x"00"),  -- REG_CTRL_USB
                              -- Bit 0:      USB Soft connect
                              -- Bit 1:      USB Suspend
                              -- Bit 2:      USB Mode
                              -- Bit 3:      USB Speed
                              -- Bit [7:4]:  Unused
      (x"000D",   x"00"),  -- REG_CTRL_USB_PROD_ID
                              -- USB Product ID
      (x"000E",   x"00"),  -- REG_CTRL_USB_STR_IDX
                              -- USB String index
      (x"000F",   x"00"),  -- REG_CTRL_USB_SERIAL_7
                              -- USB Serial number (7 downto 0 MSB to LSB)
      (x"0010",   x"00"),  -- REG_CTRL_USB_SERIAL_6
      (x"0011",   x"00"),  -- REG_CTRL_USB_SERIAL_5
      (x"0012",   x"00"),  -- REG_CTRL_USB_SERIAL_4
      (x"0013",   x"00"),  -- REG_CTRL_USB_SERIAL_3
      (x"0014",   x"00"),  -- REG_CTRL_USB_SERIAL_2
      (x"0015",   x"00"),  -- REG_CTRL_USB_SERIAL_1
      (x"0016",   x"00"),  -- REG_CTRL_USB_SERIAL_0
      (x"0017",   x"00"),  -- REG_CTRL_USB_BUF_ADD
                              -- USB Buffer Address
      (x"0018",   x"00"),  -- REG_CTRL_USB_BUF_DATA
                              -- USB Buffer Write Data
      (x"0019",   x"00"),  -- REG_CTRL_USB_BUF_WE
                              -- Bit 0:      USB Buffer Write Enable
                              -- Bit [7:1]:  Unused
      -- icap control register
      (x"001a",   x"00"),  -- REG_CTRL_ICAP_SEND,
      (x"001b",   x"00"),  -- REG_CTRL_ICAP_OPCODE_LSB,
      (x"001c",   x"00"),  -- REG_CTRL_ICAP_OPCODE_MSB,
      (x"001d",   x"00"),  -- REG_CTRL_ICAP_USERDATA_LSB,
      (x"001e",   x"00"),  -- REG_CTRL_ICAP_USERDATA_MSB,
      (x"001f",   x"00"),  -- REG_CTRL_ICAP_FALLBACK,

      -- 1 Communication with on board chips

      (x"0100",   x"00"),  -- REG_CTRL_SPI
                              -- General SPI Control
                              -- Bit 0:      Go (indicates SPI master state machine to start)
                              -- Bit 1:      Write Enable SPI Write Fifo
                              -- Bit 2:      Read Enable SPI Read Fifo
                              -- Bit 3:      Clear SPI Write Fifo
                              -- Bit 4:      Clear SPI Read Fifo
                              -- Bit 5:      Read WriteN (0: Write, 1: Read)
                              -- Bit 6:      Command with data (1: Send/receive data, 1: Don't send/receive data)
                              -- Bit 7:      Command with address (0: Don't send address, 1: Send address)
      (x"0101",   x"00"),  -- REG_CTRL_SPI_REQUEST
                              -- SPI request
                              -- 0: Basic SPI command, local flash
                              -- 1: Basic SPI command, external flash (via header)
                              -- 2: Calculate checksum of portion of SPI device data, local
                              -- 3: Calculate checksum of portion of SPI device data, external flash (via header)
                              -- 4: Copy to local flash
                              -- 5: Copy to external flash (via header)
                              -- 6: Copy to fibers
      (x"0102",   x"00"),  -- REG_CTRL_SPI_CMD
                              -- SPI command
      (x"0103",   x"00"),  -- REG_CTRL_SPI_SRC_START_ADD_HL
                              -- Spi Source Start Address
      (x"0104",   x"00"),  -- REG_CTRL_SPI_SRC_START_ADD_LH
      (x"0105",   x"00"),  -- REG_CTRL_SPI_SRC_START_ADD_LL
      (x"0106",   x"00"),  -- REG_CTRL_SPI_SRC_END_ADD_HL
                              -- Spi Source End Address
      (x"0107",   x"00"),  -- REG_CTRL_SPI_SRC_END_ADD_LH
      (x"0108",   x"00"),  -- REG_CTRL_SPI_SRC_END_ADD_LL
      (x"0109",   x"00"),  -- REG_CTRL_SPI_DEST_START_ADD_HL
                              -- Spi Destination Start Address
      (x"010A",   x"00"),  -- REG_CTRL_SPI_DEST_START_ADD_LH
      (x"010B",   x"00"),  -- REG_CTRL_SPI_DEST_START_ADD_LL
      (x"010C",   x"00"),  -- REG_CTRL_SPI_LENGTH
                              -- Length
      (x"010D",   x"00"),  -- REG_CTRL_SPI_FIFO
                              -- Data to write in SPI Fifo

      (x"010E",   x"00"),  -- REG_CTRL_I2C_CFG1
                              -- General I2C Control
                              -- Bit 0:      Go (indicates I2C master state machine to start)
                              -- Bit 1:      Write Enable I2C Write Fifo
                              -- Bit 2:      Read Enable I2C Read Fifo
                              -- Bit 3:      Clear I2C Write Fifo
                              -- Bit 4:      Clear I2C Read Fifo
                              -- Bit [7:5]:  Unused
      (x"010F",   x"00"),  -- REG_CTRL_I2C_PLL

      (x"0110",   x"00"),  -- REG_CTRL_I2C_DEV_CFG1
                              -- I2C Device Address
      (x"0111",   x"00"),  -- REG_CTRL_I2C_DEV_PLL

      (x"0112",   x"00"),  -- REG_CTRL_I2C_REG_CFG1
                              -- Register Address
      (x"0113",   x"00"),  -- REG_CTRL_I2C_REG_PLL

      (x"0114",   x"00"),  -- REG_CTRL_I2C_LENGTH_CFG1
                              -- Length
      (x"0115",   x"00"),  -- REG_CTRL_I2C_LENGTH_PLL

      (x"0116",   x"00"),  -- REG_CTRL_I2C_FIFO_CFG1
                              -- Data to write in I2C Fifo
      (x"0117",   x"00"),  -- REG_CTRL_I2C_FIFO_PLL

      -- 2 Misc

      (x"0200",   x"00"),  -- REG_CTRL_RST
                              -- Reset
                              -- Bit 0:      Reset Silicon Image Chips
                              -- Bit 1:      Reload FPGA
                              -- Bit 2:      Unused
                              -- Bit 3:      Unused
                              -- Bit 4:      Unused
                              -- Bit 5:      Unused
                              -- Bit 6:      Unused
                              -- Bit 7:      Unused

      (x"0201",   x"00"),  -- REG_CTRL_EDID_ADD
      (x"0202",   x"00"),  -- REG_CTRL_EDID_DATA
      (x"0203",   x"00"),  -- REG_CTRL_EDID_MISC
                              -- Bit 0:      Wren
                              -- Bit 1:      Unused
                              -- Bit 2:      Unused
                              -- Bit 3:      Unused
                              -- Bit 4:      Edid address bit 8
                              -- Bit 5:      Edid address bit 9
                              -- Bit 6:      Unused
                              -- Bit 7:      Unused

      (x"0204",   x"00"),  -- REG_CTRL_GEN1
                              -- Bit 0:      Unused
                              -- Bit 1:      Video module detected is a TX
                              -- Bit 2:      Start 5 second HPD timer
                              -- Bit 3:      Unused
                              -- Bit 4:      Eeprom edid status bit 0
                              -- Bit 5:      Eeprom edid status bit 1
                                             -- 00: no copy has been done since no display are connected
                                             -- 01: copy from local display has been done
                                             -- 10: copy from first remote display has been done
                                             -- 11: copy from any remote display has been done
                              -- Bit 6:      Edid status bit 0
                              -- Bit 7:      Edid status bit 1
                                             -- 00: Factory default edid
                                             -- 01: Stored/programmed edid
                                             -- 10: Local display edid, stored if no local display
                                             -- 11: Remote display edid, stored if no remote display (this mode is not enabled yet)

      -- 3 Video

      (x"0300",   x"00"),  -- REG_CTRL_H_TOTAL_H
      (x"0301",   x"00"),  -- REG_CTRL_H_TOTAL_L
      (x"0302",   x"00"),  -- REG_CTRL_H_ACTIVE_H
      (x"0303",   x"00"),  -- REG_CTRL_H_ACTIVE_L
      (x"0304",   x"00"),  -- REG_CTRL_H_BP_H
      (x"0305",   x"00"),  -- REG_CTRL_H_BP_L
      (x"0306",   x"00"),  -- REG_CTRL_H_SYNC_WIDTH_H
      (x"0307",   x"00"),  -- REG_CTRL_H_SYNC_WIDTH_L
      (x"0308",   x"00"),  -- REG_CTRL_V_TOTAL_H
      (x"0309",   x"00"),  -- REG_CTRL_V_TOTAL_L
      (x"030A",   x"00"),  -- REG_CTRL_V_ACTIVE_H
      (x"030B",   x"00"),  -- REG_CTRL_V_ACTIVE_L
      (x"030C",   x"00"),  -- REG_CTRL_V_BP
      (x"030D",   x"00"),  -- REG_CTRL_V_SYNC_WIDTH
      (x"030E",   x"00"),  -- REG_CTRL_H_FREQ_H
      (x"030F",   x"00"),  -- REG_CTRL_H_FREQ_L
      (x"0310",   x"00"),  -- REG_CTRL_V_FREQ_H
      (x"0311",   x"00"),  -- REG_CTRL_V_FREQ_L
      (x"0312",   x"00"),  -- REG_CTRL_VIDEO_MISC
                              -- Bit 0: V polarity
                              -- Bit 1: H polarity
                              -- Bit 2: Interlaced
                              -- Bit 3: Unused
                              -- Bit 4: Unused
                              -- Bit 5: Unused
                              -- Bit 6: Hdcp programmed
                              -- Bit 7: bStable (timings programmed in registers)
      (x"0313",   x"00"),  -- REG_CTRL_VIDEO_MODE

      (x"0328",   x"00"),  -- REG_CTRL_CRC

      (x"0329",   x"00"),  -- REG_CTRL_VIDEO_SOURCE_1
      (x"032A",   x"00"),  -- REG_CTRL_VIDEO_SOURCE_2
      (x"032B",   x"00"),  -- REG_CTRL_VIDEO_SOURCE_3
      (x"032C",   x"00"),  -- REG_CTRL_VIDEO_SOURCE_4
      (x"032D",   x"00"),  -- REG_CTRL_VIDEO_SOURCE_5
      (x"032E",   x"00"),  -- REG_CTRL_VIDEO_SOURCE_6

      -- 4 HDCP

      (x"0400",   x"00"),  -- REG_CTRL_HDCP_CPBL
                              -- Monitor connected on Loop-Back HDMI Tx supports HDCP
                              -- Bit 0:      PORT1
                              -- Bit 1:      Unused
                              -- Bit 2:      Unused
                              -- Bit 3:      Unused
                              -- Bit 4:      Unused
                              -- Bit 5:      Unused
                              -- Bit 6:      Unused
                              -- Bit 7:      Unused

      (x"0401",   x"00"),  -- REG_CTRL_HDCP_BCAPS_PORT1
                              -- BCaps of monitor connected to HDMI Tx

      (x"0402",   x"00"),  -- REG_CTRL_KSV_RAM_ADD
      (x"0403",   x"00"),  -- REG_CTRL_KSV_RAM_WE
      (x"0404",   x"00"),  -- REG_CTRL_KSV_RAM_DATA

      (x"0405",   x"00"),  -- REG_CTRL_NEW_BKSV

      (x"0406",   x"00"),  -- REG_CTRL_HDCP_IN_ENABLED

      (x"0407",   x"00"),  -- REG_CTRL_TOGGLE_HPD_IN

      -- 5 Communication with Packet Router

      (x"0500",   x"00"),  -- REG_CTRL_TX_PACKET_7
      (x"0501",   x"00"),  -- REG_CTRL_TX_PACKET_6
      (x"0502",   x"00"),  -- REG_CTRL_TX_PACKET_5
      (x"0503",   x"00"),  -- REG_CTRL_TX_PACKET_4
      (x"0504",   x"00"),  -- REG_CTRL_TX_PACKET_3
      (x"0505",   x"00"),  -- REG_CTRL_TX_PACKET_2
      (x"0506",   x"00"),  -- REG_CTRL_TX_PACKET_1
      (x"0507",   x"00"),  -- REG_CTRL_TX_PACKET_0
      (x"0508",   x"00"),  -- REG_CTRL_TX_PACKET_ADD_CAT
                              -- Bit [3:0]:  Packet category
                              -- Bit [7:4]:  Packet address
      (x"0509",   x"00"),  -- REG_CTRL_TX_PACKET_GEN
                              -- Bit [1:0]:  Packet priority
                              -- Bit [6:2]:  Unused
                              -- Bit 7:      TX packet fifo Wren

      (x"050A",   x"00"),  -- REG_CTRL_RX_PACKET_GEN_GTP0
                              -- Bit 0:      Unused
                              -- Bit 1:      Unused
                              -- Bit 2:      Unused
                              -- Bit 3:      Unused
                              -- Bit 4:      Unused
                              -- Bit 5:      Unused
                              -- Bit 6:      Unused
                              -- Bit 7:      RX packet fifo Read
      (x"050B",   x"00"),  -- REG_CTRL_RX_PACKET_GEN_GTP1
      (x"050C",   x"00"),  -- REG_CTRL_RX_PACKET_GEN_GTP2
      (x"050D",   x"00"),  -- REG_CTRL_RX_PACKET_GEN_GTP3

      -- 6 Packet Router

      (x"0600",   x"00"),  -- REG_CTRL_ROUTER_TESTER_A_TX_1
      (x"0601",   x"00"),  -- REG_CTRL_ROUTER_TESTER_A_TX_2
      (x"0602",   x"00"),  -- REG_CTRL_ROUTER_TESTER_A_TX_3
      (x"0603",   x"00"),  -- REG_CTRL_ROUTER_TESTER_A_TX_4_LL
      (x"0604",   x"00"),  -- REG_CTRL_ROUTER_TESTER_A_TX_4_LH
      (x"0605",   x"00"),  -- REG_CTRL_ROUTER_TESTER_A_TX_4_HL
      (x"0606",   x"00"),  -- REG_CTRL_ROUTER_TESTER_A_TX_4_HH
      (x"0607",   x"00"),  -- REG_CTRL_ROUTER_TESTER_A_RX_1

      (x"0608",   x"00"),  -- REG_CTRL_ROUTER_TESTER_B_TX_1
      (x"0609",   x"00"),  -- REG_CTRL_ROUTER_TESTER_B_TX_2
      (x"060A",   x"00"),  -- REG_CTRL_ROUTER_TESTER_B_TX_3
      (x"060B",   x"00"),  -- REG_CTRL_ROUTER_TESTER_B_TX_4_LL
      (x"060C",   x"00"),  -- REG_CTRL_ROUTER_TESTER_B_TX_4_LH
      (x"060D",   x"00"),  -- REG_CTRL_ROUTER_TESTER_B_TX_4_HL
      (x"060E",   x"00"),  -- REG_CTRL_ROUTER_TESTER_B_TX_4_HH
      (x"060F",   x"00"),  -- REG_CTRL_ROUTER_TESTER_B_RX_1

      (x"0610",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_0_GENERAL_1
      (x"0611",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_0_CHANNEL_LL
      (x"0612",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_0_CHANNEL_LH
      (x"0613",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_0_CHANNEL_HL
      (x"0614",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_0_CHANNEL_HH

      (x"0615",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_1_GENERAL_1
      (x"0616",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_1_CHANNEL_LL
      (x"0617",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_1_CHANNEL_LH
      (x"0618",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_1_CHANNEL_HL
      (x"0619",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_1_CHANNEL_HH

      (x"061A",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_2_GENERAL_1
      (x"061B",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_2_CHANNEL_LL
      (x"061C",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_2_CHANNEL_LH
      (x"061D",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_2_CHANNEL_HL
      (x"061E",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_2_CHANNEL_HH

      (x"061F",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_3_GENERAL_1
      (x"0620",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_3_CHANNEL_LL
      (x"0621",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_3_CHANNEL_LH
      (x"0622",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_3_CHANNEL_HL
      (x"0623",   x"00"),  -- REG_CTRL_ROUTER_BRIDGE_3_CHANNEL_HH

      -- 7 Audio
      
      (x"0700",   x"00"),  -- REG_CTRL_AUDIO_NOT_USED0,
      (x"0701",   x"00"),  -- REG_CTRL_AUDIO_NOT_USED1,
      (x"0702",   x"00"),  -- REG_CTRL_AUDIO_AUX_DAC_DATA,
      (x"0703",   x"00"),  -- REG_CTRL_AUDIO_AUX_DAC_ADDR,
      (x"0704",   x"00")   -- REG_CTRL_AUDIO_AUX_DAC_WE
   );

   -- Cpu status registers address
   constant STAT_REG_ADD : TCpuStatRegAddArray :=
   (
      -- 0 CPU

      x"4000",    -- REG_STAT_LEVEL_INT_CPU
                     -- CPU Interrupt status register (level interrupts)

      x"4001",    -- REG_STAT_USB
                     -- USB Status
                     -- Bit 0:      Usb Busy
                     -- Bit 1:      Crc 16 Error
                     -- Bit 2:      Crc 5 Error
                     -- Bit 3:      Rx Ack Timeout
                     -- Bit 4:      Usb Reset
                     -- Bit 5:      Output Full
                     -- Bit 6:      Ready To Send
                     -- Bit 7:      Unused
      x"4002",    -- REG_STAT_USB_BUF_DATA
                     -- USB Buffer Data Read

      x"4003",    -- REG_STAT_CPU_ROM_EXTRA_DATA_PATCH
                     -- ERomTX/RX data bits 15:14 (patch to simplify recompile of uC, prog on 16 bits instead of 14)
                     -- Bit[1:0]:   ERomTX/RX data bits 15:14
                     -- Bit[7:2]:   Unused

      x"4004",    -- REG_STAT_FPGA_BUILD_NUMBER_LSB
      x"4005",    -- REG_STAT_FPGA_BUILD_NUMBER_MSB

      x"4006",    -- REG_STAT_ICAP_BUSY,
      x"4007",    -- REG_STAT_ICAP_USERDATA_LSB,
      x"4008",    -- REG_STAT_ICAP_USERDATA_MSB,

      -- 1 Communication with on board chips

      x"4100",    -- REG_STAT_SPI_FIFO
                     -- Data read from I2C Fifo
      x"4101",    -- REG_STAT_SPI_CHECKSUM

      x"4102",    -- REG_STAT_I2C_FIFO_CFG1
                     -- Data read from I2C Fifo
      x"4103",    -- REG_STAT_I2C_FIFO_PLL

      -- 2 Misc

      x"4200",    -- REG_STAT_LP_HPD
                     -- Hot Plug Detect on Loop-Back HDMI Tx
                     -- Bit 0:      CFG1
                     -- Bit 1:      CFG2 Unused
                     -- Bit 2:      CFG3 Unused
                     -- Bit 3:      CFG4 Unused
                     -- Bit 4:      CFG5 Unused
                     -- Bit 5:      CFG6 Unused
                     -- Bit 6:      CFG7 Unused
                     -- Bit 7:      CFG8 Unused

      x"4201",    -- REG_STAT_EDID_DATA

      x"4202",    -- REG_STAT_GEN1
                     -- General purpose status
                     -- Bit 0:      Timer 0 has expired
                     -- Bit 1:      Timer 1 has expired
                     -- Bit 2:      Unused
                     -- Bit 3:      Unused
                     -- Bit 4:      Unused
                     -- Bit 5:      Unused
                     -- Bit 6:      Edid mode to set bit 0
                     -- Bit 7:      Edid mode to set bit 1
                                    -- 00: Factory default edid
                                    -- 01: Stored/programmed edid
                                    -- 10: Local display edid, stored if no local display
                                    -- 11: Remote display edid, stored if no remote display (this mode is not enabled yet)

      x"4203",    -- REG_STAT_BOARD_ID
                     -- Bits [3:0]: Video board id
                     -- Bits [7:4]: Aux board id

      -- 3 Video

      x"4300",     -- REG_STAT_H_TOTAL_H
      x"4301",     -- REG_STAT_H_TOTAL_L
      x"4302",     -- REG_STAT_H_ACTIVE_H
      x"4303",     -- REG_STAT_H_ACTIVE_L
      x"4304",     -- REG_STAT_H_BP_H
      x"4305",     -- REG_STAT_H_BP_L
      x"4306",     -- REG_STAT_H_SYNC_WIDTH_H
      x"4307",     -- REG_STAT_H_SYNC_WIDTH_L
      x"4308",     -- REG_STAT_V_TOTAL_H
      x"4309",     -- REG_STAT_V_TOTAL_L
      x"430A",     -- REG_STAT_V_ACTIVE_H
      x"430B",     -- REG_STAT_V_ACTIVE_L
      x"430C",     -- REG_STAT_V_BP
      x"430D",     -- REG_STAT_V_SYNC_WIDTH
      x"430E",     -- REG_STAT_H_FREQ_H
      x"430F",     -- REG_STAT_H_FREQ_L
      x"4310",     -- REG_STAT_V_FREQ_H
      x"4311",     -- REG_STAT_V_FREQ_L
      x"4312",     -- REG_STAT_VIDEO_MISC
                     -- Bit 0: V polarity
                     -- Bit 1: H polarity
                     -- Bit 2: Interlaced
                     -- Bit 3: Unused
                     -- Bit 4: Unused
                     -- Bit 5: Unused
                     -- Bit 6: Hdcp required
                     -- Bit 7: bStable (active video in and timings received on fiber are stable)

      x"4313",     -- REG_STAT_CRC_DVO_ERR_CNT_LL
      x"4314",     -- REG_STAT_CRC_DVO_ERR_CNT_LH
      x"4315",     -- REG_STAT_CRC_DVO_ERR_CNT_HL
      x"4316",     -- REG_STAT_CRC_DVO_ERR_CNT_HH

      x"4317",     -- REG_STAT_CRC_WRITE_ERR_CNT_LL
      x"4318",     -- REG_STAT_CRC_WRITE_ERR_CNT_LH
      x"4319",     -- REG_STAT_CRC_WRITE_ERR_CNT_HL
      x"431A",     -- REG_STAT_CRC_WRITE_ERR_CNT_HH

      x"431B",     -- REG_STAT_CRC_READ_ERR_CNT_LL
      x"431C",     -- REG_STAT_CRC_READ_ERR_CNT_LH
      x"431D",     -- REG_STAT_CRC_READ_ERR_CNT_HL
      x"431E",     -- REG_STAT_CRC_READ_ERR_CNT_HH

      x"431F",     -- REG_STAT_CRC_DVO_FRM_CNT_LL
      x"4320",     -- REG_STAT_CRC_DVO_FRM_CNT_LH
      x"4321",     -- REG_STAT_CRC_DVO_FRM_CNT_HL
      x"4322",     -- REG_STAT_CRC_DVO_FRM_CNT_HH

      x"4323",     -- REG_STAT_CRC_WRITE_FRM_CNT_LL
      x"4324",     -- REG_STAT_CRC_WRITE_FRM_CNT_LH
      x"4325",     -- REG_STAT_CRC_WRITE_FRM_CNT_HL
      x"4326",     -- REG_STAT_CRC_WRITE_FRM_CNT_HH

      x"4327",     -- REG_STAT_CRC_READ_FRM_CNT_LL
      x"4328",     -- REG_STAT_CRC_READ_FRM_CNT_LH
      x"4329",     -- REG_STAT_CRC_READ_FRM_CNT_HL
      x"432A",     -- REG_STAT_CRC_READ_FRM_CNT_HH

      x"432B",     -- REG_STAT_CRC_VAL_LL
      x"432C",     -- REG_STAT_CRC_VAL_LH
      x"432D",     -- REG_STAT_CRC_VAL_HL
      x"432E",     -- REG_STAT_CRC_VAL_HH

      -- 4 HDCP

      x"4400",    -- REG_STAT_KSV_RAM_DATA

      -- 5 Communication with Packet Router

      x"4500",     -- REG_STAT_TX_PACKET_GEN
                     -- Bit 0:      Unused
                     -- Bit 1:      Unused
                     -- Bit 2:      Unused
                     -- Bit 3:      Unused
                     -- Bit 4:      Unused
                     -- Bit 5:      Unused
                     -- Bit 6:      Unused
                     -- Bit 7:      TX packet fifo ready

      x"4501",     -- REG_STAT_RX_PACKET_7_GTP0
      x"4502",     -- REG_STAT_RX_PACKET_7_GTP1
      x"4503",     -- REG_STAT_RX_PACKET_7_GTP2
      x"4504",     -- REG_STAT_RX_PACKET_7_GTP3
      x"4505",     -- REG_STAT_RX_PACKET_6_GTP0
      x"4506",     -- REG_STAT_RX_PACKET_6_GTP1
      x"4507",     -- REG_STAT_RX_PACKET_6_GTP2
      x"4508",     -- REG_STAT_RX_PACKET_6_GTP3
      x"4509",     -- REG_STAT_RX_PACKET_5_GTP0
      x"450A",     -- REG_STAT_RX_PACKET_5_GTP1
      x"450B",     -- REG_STAT_RX_PACKET_5_GTP2
      x"450C",     -- REG_STAT_RX_PACKET_5_GTP3
      x"450D",     -- REG_STAT_RX_PACKET_4_GTP0
      x"450E",     -- REG_STAT_RX_PACKET_4_GTP1
      x"450F",     -- REG_STAT_RX_PACKET_4_GTP2
      x"4510",     -- REG_STAT_RX_PACKET_4_GTP3
      x"4511",     -- REG_STAT_RX_PACKET_3_GTP0
      x"4512",     -- REG_STAT_RX_PACKET_3_GTP1
      x"4513",     -- REG_STAT_RX_PACKET_3_GTP2
      x"4514",     -- REG_STAT_RX_PACKET_3_GTP3
      x"4515",     -- REG_STAT_RX_PACKET_2_GTP0
      x"4516",     -- REG_STAT_RX_PACKET_2_GTP1
      x"4517",     -- REG_STAT_RX_PACKET_2_GTP2
      x"4518",     -- REG_STAT_RX_PACKET_2_GTP3
      x"4519",     -- REG_STAT_RX_PACKET_1_GTP0
      x"451A",     -- REG_STAT_RX_PACKET_1_GTP1
      x"451B",     -- REG_STAT_RX_PACKET_1_GTP2
      x"451C",     -- REG_STAT_RX_PACKET_1_GTP3
      x"451D",     -- REG_STAT_RX_PACKET_0_GTP0
      x"451E",     -- REG_STAT_RX_PACKET_0_GTP1
      x"451F",     -- REG_STAT_RX_PACKET_0_GTP2
      x"4520",     -- REG_STAT_RX_PACKET_0_GTP3
      x"4521",     -- REG_STAT_RX_PACKET_ADD_CAT_GTP0
                     -- Bit [3:0]:  Packet category
                     -- Bit [7:4]:  Packet address
      x"4522",     -- REG_STAT_RX_PACKET_ADD_CAT_GTP1
      x"4523",     -- REG_STAT_RX_PACKET_ADD_CAT_GTP2
      x"4524",     -- REG_STAT_RX_PACKET_ADD_CAT_GTP3
      x"4525",     -- REG_STAT_RX_PACKET_GEN_GTP0
                     -- Bit [6:0]:  Unused
                     -- Bit 7:      RX packet fifo empty
      x"4526",     -- REG_STAT_RX_PACKET_GEN_GTP1
      x"4527",     -- REG_STAT_RX_PACKET_GEN_GTP2
      x"4528",     -- REG_STAT_RX_PACKET_GEN_GTP3

      -- 6 Packet Router

      x"4600",     -- REG_STAT_ROUTER_TESTER_A_FRAME_COUNT_LL
      x"4601",     -- REG_STAT_ROUTER_TESTER_A_FRAME_COUNT_LH
      x"4602",     -- REG_STAT_ROUTER_TESTER_A_FRAME_COUNT_HL
      x"4603",     -- REG_STAT_ROUTER_TESTER_A_FRAME_COUNT_HH

      x"4604",     -- REG_STAT_ROUTER_TESTER_A_ERROR_COUNT_LL
      x"4605",     -- REG_STAT_ROUTER_TESTER_A_ERROR_COUNT_LH
      x"4606",     -- REG_STAT_ROUTER_TESTER_A_ERROR_COUNT_HL
      x"4607",     -- REG_STAT_ROUTER_TESTER_A_ERROR_COUNT_HH

      x"4608",     -- REG_STAT_ROUTER_TESTER_A_CRC_LL
      x"4609",     -- REG_STAT_ROUTER_TESTER_A_CRC_LH
      x"460A",     -- REG_STAT_ROUTER_TESTER_A_CRC_HL
      x"460B",     -- REG_STAT_ROUTER_TESTER_A_CRC_HH

      x"460C",     -- REG_STAT_ROUTER_TESTER_B_FRAME_COUNT_LL
      x"460D",     -- REG_STAT_ROUTER_TESTER_B_FRAME_COUNT_LH
      x"460E",     -- REG_STAT_ROUTER_TESTER_B_FRAME_COUNT_HL
      x"460F",     -- REG_STAT_ROUTER_TESTER_B_FRAME_COUNT_HH

      x"4610",     -- REG_STAT_ROUTER_TESTER_B_ERROR_COUNT_LL
      x"4611",     -- REG_STAT_ROUTER_TESTER_B_ERROR_COUNT_LH
      x"4612",     -- REG_STAT_ROUTER_TESTER_B_ERROR_COUNT_HL
      x"4613",     -- REG_STAT_ROUTER_TESTER_B_ERROR_COUNT_HH

      x"4614",     -- REG_STAT_ROUTER_TESTER_B_CRC_LL
      x"4615",     -- REG_STAT_ROUTER_TESTER_B_CRC_LH
      x"4616",     -- REG_STAT_ROUTER_TESTER_B_CRC_HL
      x"4617"      -- REG_STAT_ROUTER_TESTER_B_CRC_HH
   );

   -- Cpu interrupt registers address
   constant INT_REG_ADD : TCpuIntRegAddArray :=
   (
      -- 0 CPU

      x"8000",    -- REG_INT_EDGE_INT_CPU
                     -- CPU Interrupt status register (edge interrupts)

      x"8001",    -- REG_INT_MISC
                     -- USB status register that ca be cleared by uC
                     -- Bit 0:      USB Buf Empty
                     -- Bit 1:      Replace stored edid in eeprom
                     -- Bit [7:2]:  Unused

      x"8002",    -- REG_INT_EDGE_INT_CPU_OCCURED
                     -- Edge interrupt occured, will not be cleared by uC
      x"8003",    -- REG_INT_LEVEL_INT_CPU_OCCURED
                     -- Level interrupt occured, will not be cleared by uC

      -- 1 Communication with on board chips

      x"8100",    -- REG_INT_SPI
                     -- SPI status register that can be cleared by uC
                     -- Bit 0:      I2C transaction done
                     -- Bit [7:1]:  Unused

      x"8101",    -- REG_INT_I2C_CFG1
                     -- I2C status register that can be cleared by uC
                     -- Bit 0:      I2C transaction done
                     -- Bit 1:      I2C transaction error
                     -- Bit [7:2]:  Unused
      x"8102"     -- REG_INT_I2C_PLL
   );

end package;